#ifndef _WRITEMIME_H_
#define _WRITEMIME_H_
/* @(#$Id: writemime.h,v 1.26 2005/01/27 17:06:15 dockes Exp $  (C) 2002 Jean-Francois Dockes */

#include <list>
#include <string>

#ifndef NO_NAMESPACES
using std::list;
using std::string;
#endif // NO_NAMESPACES

/// Mime message encoding.
namespace WriteMime 
{

// Header field parameter (ie: ';filename="toto"')
class HeaderFieldParam {
  char *m_name;
  char *m_value;
  int   m_size;
  void reset() {
	if (m_name) {free(m_name); m_name = 0;}
	if (m_value) {free(m_value); m_value = 0;}
	m_size = 0;
  }
  void buildFromFriend(const HeaderFieldParam& rh);

 public:
  HeaderFieldParam(const char *nm, const char *value)
	: m_name(strdup(nm)), m_value(strdup(value)), m_size(0)	{}
  HeaderFieldParam(const HeaderFieldParam& rh) {buildFromFriend(rh);}
  ~HeaderFieldParam() {reset();}
  HeaderFieldParam& operator=(const HeaderFieldParam& l);
  // returns the size -without any terminating 0- Print() WILL overshoot
  // size by 1
  int size();
  // Format text into buffer, which should be size() + 1 bytes
  int print(char *d);
};

// Header field, like 'Content-Disposition: attachment; filename="toto"'
class HeaderField {
  char *m_name;
  char *m_value;
  int   m_size;
  list<HeaderFieldParam> m_params;

  void reset() {
	if (m_name) {free(m_name); m_name = 0;}
	if (m_value) {free(m_value); m_value = 0;}
	m_size = 0;
  }
  void buildFromFriend(const HeaderField &rh);
 public:
  HeaderField(const char *nm, const char *value) 
	: m_name(strdup(nm)), m_value(strdup(value)), m_size(0) {}
  HeaderField(const HeaderField &rh) {buildFromFriend(rh);}
  ~HeaderField() {reset();}
  HeaderField &operator=(const HeaderField &rh);
  int addparam(const char *pnm, const char *pval);
  // Return size, -without any terminating 0- print() WILL overshoot size by 1
  int size();
  // Format text into buffer, which should be size() + 1 bytes
  int print(char *d);
};



/// virtual base class for Mime entities.
class Entity {
 public:
  /// Media types. Subtypes are left as free form text strings.
  enum type {TEXT,IMAGE,AUDIO,VIDEO,APPLICATION,MULTIPART,MESSAGE};
  /// Content disposition. Note: the more natural INLINE name causes
  /// too much problems with headers that #define an INLINE
  enum disposition {DISP_ATTACHMENT, DISP_INLINE};
  /// Content transfer encoding. 
  enum encoding {ENC_7BIT, ENC_8BIT, ENC_BINARY, ENC_QP, ENC_BASE64};

 private:
  char                *m_subtype;
  char                *m_error;

 protected:
  type                 m_type;
  encoding             m_encoding;
  list<HeaderField>    m_hf;
  int                  m_size;
  char                *m_fmt;  // Formatted buffer

 public:
  Entity() : m_subtype(strdup("plain")), m_error(0), 
	m_type(TEXT),  m_encoding(ENC_BINARY), m_size(0), m_fmt(0) {}
  virtual ~Entity() {
	if (m_subtype) {free(m_subtype); m_subtype = 0;}
	if (m_error) {free(m_error); m_error = 0;}
	if (m_fmt) {free(m_fmt); m_fmt = 0;}
  }

  /// Set current error string (mainly used by sendmail() or internally).
  virtual void seterror(const char *s);
  /// Return current error string.
  virtual const char *geterror() {return m_error;}
  /// Return current encoding.
  virtual encoding getencoding() {return m_encoding;}

  /// Add header field.
  virtual void addHeaderField(const HeaderField &f);
  virtual int size() = 0;
  virtual const char *format() = 0;
  virtual int format(char *buf) = 0;

 protected:
  virtual int addCteHeader();
  virtual int allocFmt();
  virtual int formatHeaders(char *buf);
  virtual int formatHeaders();
  virtual void settype(type tp) {m_type = tp;}
  virtual int setsubtype(const char *stp);
  virtual const char *getsubtype() {return m_subtype;}
};

/// A DiscreteEntity represents a MIME entity with a discrete media
/// type, it can be used either as the base for a simple message or as
/// a part of a CompositeEntity.
class DiscreteEntity : public virtual Entity {
  char          *m_body;
  int            m_blen;
  encoding       m_textencodepref;
  string    m_charset;

  int addCtpHeader();

 public:
  DiscreteEntity() : m_body(0), m_blen(0), m_charset("") {}
  virtual ~DiscreteEntity() {if (m_body) {free(m_body); m_body = 0;}}

  /// Set encoding preference for 8 bit text: either 8bit or quoted-printable
  virtual void setTextEncodePref(encoding ec);
  
  /// Set the character set for a TEXT body.
  virtual void setCharSet(const char *cs) {m_charset = cs;}

  /// Create body from data buffer.
  ///
  /// If the type is TEXT, the buffer transfer-encoding will be 7BIT
  /// if the data is ascii, or else either QP or 8BIT (depending on
  /// textencodepref).
  ///
  /// If the type is not TEXT, the buffer will be encoded in BASE64.
  ///
  /// @param bp points to body text (no need for final 0, may be binary).
  /// @param blen is the input buffer size.
  /// @param tp MIME type for data.
  /// @param stp MIME subtype.
  /// @return -1 for error, 0 else.
  virtual int setBody(const char *bp, int blen, type tp = TEXT, 
		  const char *stp = "plain");

  /// Create body by attaching file (will be encoded in BASE64).
  /// @param filename file name.
  /// @param tp MIME type for data.
  /// @param stp MIME subtype.
  /// @return -1 for error, 0 else.
  virtual int setBodyAttach(const char *filename, type tp = APPLICATION,
					const char *stp = "octet-stream");
  /// Compute storage size, including final 0 byte.
  virtual int size();
  /// Return malloc'd, 0-terminated buffer of formatted contents.
  virtual const char *format();
  /// Format into caller buffer, which must be at least size() bytes wide.
  virtual int         format(char *buf);
};

/// The CompositeEntity class is used to represent MIME entities with
/// a composite media type, which will typically have multiple body
/// parts.  
///
/// The class only supports media of MULTIPART type with ALTERNATIVE
/// subtype for now. In addition it will be only accept discrete
/// entities as body parts, which is probably an unnecessary
/// restriction.
class CompositeEntity : public virtual Entity {
  list<Entity *>   m_parts;
  int              m_size;
  string      m_boundary;
 public:
  CompositeEntity(type tp, const char *stp);
  virtual ~CompositeEntity();
  /// Add a body part.
  virtual int addPart(DiscreteEntity *part);
  /// Compute storage size, including final 0 byte.
  virtual int size();
  /// Return malloc'd, 0-terminated buffer of formatted contents.
  virtual const char *format();
  /// Format into caller buffer, which must be at least size() bytes wide.
  virtual int  format(char *buf);
};

// Recipient address storage
class Recipient {
  string official;
  string nickname;
 public:
  Recipient(const char *off, const char *nick) 
	: official(off), nickname(nick?nick:"") 
	{}
  friend class Message;
  friend class SimpleMessage;
  friend class CompositeMessage;
};

/// Pure virtual base class for email messages.
class Message : public virtual Entity {
 protected:
  list<Recipient> to;
  list<Recipient> cc;
  list<Recipient> bcc;
  static const char *recpth[3];
  list<Recipient> *recptlists[3];

  void setRecipientHeaders();

 public:
  /// Destination address types.
  enum recpt_type {TO = 0, CC = 1, BCC = 2};

  Message();
  virtual ~Message() {}

  /// return 0-terminated formatted message text
  virtual const char *format() = 0;

  /// Add recipient to one of the recipient lists.
  /// @param official will be placed inside <>.
  /// @param nick will be printed before the formal address.
  /// @param tp defines what kind of recipient this is (to, cc, bcc).
  virtual int addRecipient(const char *official, const char *nick = 0, 
			   recpt_type tp = TO);

  /// Parse a user-written recipient string and add recipients.  The
  /// purpose of this routine is to help a program interpret text
  /// cut-and-pasted by a user into a Gui text field. This has little
  /// to do with rfc822 and does not even try to pretend to support
  /// everything that could get into an address string. 
  ///
  /// The function *CAN* fail if the syntax is bad (ie: unbalanced '<').
  ///
  /// It supports comma-separated lists of addresses, where
  /// each address can contain comments '(bla)', and look like a
  /// simple address (dockes or dockes@my.dom.ain), or a full name +
  /// official address: jean-francois dockes <dockes>
  ///
  /// The function also supports quoting with double-quotes, for
  /// exemple to avoid comma interpretation inside the full name part.
  ///
  /// Exemple of a list of 2 addresses: @par
  ///  dockes, "Smith, Bill" <bill@smith.org> (The Bill Smith)
  /// 
  ///  @param in the string to be parsed
  ///  @param tp defines the target recipient list (to, cc, or bcc).
  ///  @return 0 for success, -1 if parsing failed. Sets the message
  ///     error string in this case.
  virtual int parseAddRecipients(const char *in, recpt_type tp = TO);

  /// Return recipient list as a C string array.
  /// @return an array of pointers to 0-terminated C strings. The last
  ///   entry in the array is null. The caller should free the array
  ///   of pointers when done, but *not* the individual strings.
  virtual const char **getRecipients();

  /// Set the subject field value.
  virtual int setSubject(const char *subject);
  /// Set the 'From' field value.
  virtual int setFrom(const Recipient &from);
};

/// A simple email message is a discrete MIME entity plus
/// email-specific headers.
/// Use the DiscreteEntity setBody() or setBodyAttach() methods to
/// create the body. Use the Message methods to set email headers.
class SimpleMessage : public Message, public DiscreteEntity {
 public:
  SimpleMessage();
  /// return 0-terminated formatted message text
  virtual const char *format();
};

/// A composite message is a composite entity plus email-specific headers.
/// Use the CompositeEntity methods to add body parts, which are each
/// to be created as an Entity.Use the Message methods to set the
/// email headers.
class CompositeMessage : public Message, public CompositeEntity {
 public:
  CompositeMessage(type tp, const char *stp);
  /// return 0-terminated formatted message text
  virtual const char *format();
};

/// Read file into memory.
/// @param fname file name
/// @param outlen returns the file/output buffer size
/// @return A buffer holding the raw file data, which should be freed by the
///     caller.
extern char *readFile(const char *fname, int *outlen);

/// Encode input buffer to base64 format.
/// @param from input buffer.
/// @param fromlen input buffer size.
/// @param tolen returns the generated data size, not including the final 0
/// @param ll Output line length in terms of input chars. 54 input
///           bytes -> 72 output
/// @return Encoded buffer. The memory is allocated through malloc and 
///   should be freed by the caller. The buffer has a final 0 byte.
extern char *base64Encode(const char *from, int fromlen , int *tolen, 
						  int ll = 54);

/// Encode file to base64 format.
/// @param fname the file name
/// @param outlen the encoded buffer size, not including the final 0
/// @return Encoded buffer. The memory is allocated through malloc and 
///   should be freed by the caller. The buffer has a final 0 byte.
extern char *base64EncodeFile(const char *fname, int *outlen);

/// Encode input buffer to quoted-printable format.
/// @param from input buffer.
/// @param fromlen input buffer size.
/// @param tolen returns the generated data size, not including the final 0
/// @return Encoded buffer. The memory is allocated through malloc and 
///   should be freed by the caller. The buffer has a final 0 byte.
extern char *qpEncode(const char *from, int fromlen, int *tolen);

#ifdef WITH_SENDMAIL
/// Execute sendmail to send message.
/// Extracts the complete list of recipients and executes sendmail, like
/// '/usr/sbin/sendmail -i -- [recipient list]'. @par 
/// The default program to be executed is defined by the SENDMAIL_PROG
/// preprocessor variable (default: /usr/sbin/sendmail). This can be 
/// overridden either by defining the variable at compile time, or, at 
/// run time through an environment variable: WRITEMIME_SENDMAIL_PROG
/// 
/// @param msg the message to send. Address lists for sendmail
///   execution will be generated from the to, cc, and bcc lists.
/// @return 0 for success, -1 for failure.
extern int sendmail(Message *msg);
#endif /* WITH_SENDMAIL */

} // End namespace WriteMime

#endif /* _WRITEMIME_H_ */
