#ifndef lint
static char rcsid [] = "@(#$Id: writemime.cpp,v 1.34 2005/01/27 17:06:15 dockes Exp $  (C) 2002 Jean-Francois Dockes";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "writemime.h"

#ifdef WITH_SENDMAIL
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#endif // WITH_SENDMAIL

#ifdef USING_DMALLOC
#include "/usr/local/include/dmalloc.h"
#endif /* USING_DMALLOC */

#ifndef NO_NAMESPACES
namespace WriteMime {
using namespace std;
#endif // NO_NAMESPACES

#ifndef freeZ
#define freeZ(X) {if (X) {free(X);X=0;}}
#endif 

/////////// Base64 encoding
static const char base64chars[] = {
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
  'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
  'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
  'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
  '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/',
};

static inline void 
base64EncodeTriplet(unsigned char *to_p, const unsigned char *in_p)
{
  *to_p++ = base64chars[(in_p[0] & 0xfc) >> 2];
  *to_p++ = base64chars[((in_p[0] & 3) << 4) | ((in_p[1] & 0xf0)>>4)];
  *to_p++ = base64chars[((in_p[1] & 0x0f) << 2) | ((in_p[2] & 0xc0)>>6)];
  *to_p++ = base64chars[(in_p[2] & 0x3f)];
}

/// @param ll Output line length in terms of input chars. 54 input
/// bytes -> 72 output
char *base64Encode(const char *from, int fromlen, int *tolen, int ll)
{
  // Size computation: allow for 5 chars expansion for last char (char
  // -> 2 chars +2 '=' + crlf, + ending 0. All the rest expands from 3
  // to 4 plus 1 crlf every LL input chars
  int m_tolen = (4 * fromlen) / 3 + 2 * fromlen / ll + 6 + 1 + 20;

  unsigned char *to = (unsigned char *)malloc(m_tolen);
  if (to == 0)
	return 0;

  unsigned char *to_p = to;
  unsigned char *ufrom = (unsigned char *)from;
  int beforepad = fromlen / 3 * 3;
  for (int ii = 0 ; ii < beforepad; ii += 3) {
	base64EncodeTriplet(to_p, ufrom + ii);
	to_p += 4;
	if (ii && ((ii+3) % ll == 0)) {
	  *to_p++ = '\r';
	  *to_p++ = '\n';
	}
  }

  // Residual
  int resid = fromlen % 3;
  if (resid) {
	unsigned char t[3] = {0,0,0};
	memcpy(t, ufrom + fromlen - resid, resid);
	base64EncodeTriplet(to_p, t);
	if (resid == 1) {
	  to_p += 2;
	  *to_p++ = '=';
	  *to_p++ = '=';
	} else {
	  to_p += 3;
	  *to_p++ = '=';
	}	  
	*to_p++ = '\r';
	*to_p++ = '\n';
  } else if (fromlen % ll) {
	*to_p++ = '\r';
	*to_p++ = '\n';
  }

  *tolen = to_p - to;
  *to_p++ = 0;
  //  fprintf(stderr, "base64Encode: allocated %d, used %d\n", m_tolen,
  //		  to_p - to);
  return (char *)to;
}

char *qpEncode(const char *sfrom, int fromlen, int *tolen)
{
  unsigned char *from = (unsigned char *)sfrom;
  unsigned char hex[] = "0123456789ABCDEF";

  // Allocate output buffer: max size is 3 bytes per input byte + 3
  // bytes of soft line break every 75 output bytes.
  int m_tolen = 3 * fromlen + 3 * ((3*fromlen) / 75) + 20;
  unsigned char *to_base = (unsigned char *)malloc(m_tolen);
  if (to_base == 0)
	return 0;

  unsigned char *to = to_base;
  unsigned char *line_base = to;
  while (fromlen--) {
	// Escape "^From " and "^."
	if (to - line_base == 5 && !memcmp(line_base, "From ", 5)) {
	  strcpy((char *)line_base, "=46rom ");
	  to += 2;
	} else if (to - line_base == 1 && line_base[0] == '.') {
	  strcpy((char *)line_base, "=2E");
	  to += 2;
	}

	// Process next char
	unsigned char c = *from++;

	// Hard Line break ?
	if (c == '\n' || (c == '\r' && fromlen && *from == '\n')) {
	  *to++ = '\r';
	  *to++ = '\n';
	  if (*from == '\n') {
		fromlen--;
		from++;
	  }
	  line_base = to;
	  continue;
	}
	// Need quoting ?
	int needquote = 1;
	if ((33 <= c && c <= 60) || (62 <= c && c <= 126) || 
		(c == 32 && fromlen && *from != '\r' && *from != '\n'))
	  needquote = 0;

	// Need soft line break ?
	if ( (needquote  && (to - line_base >= 73)) || 
		 (!needquote && (to - line_base >= 75)) ) {
	  memcpy(to, "=\r\n", 3);
	  to += 3;
	  line_base = to;
	}

	if (needquote) {
	  *to++ = '=';
	  *to++ = hex[c >> 4];
	  *to++ = hex[c & 0xf];
	} else {
	  *to++ = c;
	}
  }

  // Final soft line break if needed. Not clear from the rfc that
  // this is actually needed, but it really seems saner
  if (to == to_base || to[-1] != '\n') {
	  memcpy(to, "=\r\n", 3);
	  to += 3;
  }	
  *tolen = to - to_base;
  *to++ = 0;
  //  fprintf(stderr, "base64Encode: allocated %d, used %d\n", m_tolen,
  //		  to - to_base);
  return (char *)to_base;
}

// Read file into malloced memory buffer
char *readFile(const char *fname, int *sizep)
{
  FILE *fp = 0;
  char *cp = 0;
  int size = 0;
  fp = fopen(fname, "rb");
  if (fp == 0) {
	perror("fopen");
	goto out;
  }
  fseek(fp, 0, SEEK_END);
  size = ftell(fp);
  rewind(fp);
  cp = (char *)malloc(size);
  if (cp == 0) {
	perror("malloc");
	goto out;
  }
  if (fread(cp, 1, size, fp) != (size_t)size) {
	perror("fread");
	goto out;
  }
 out: 
  if (fp)
	fclose(fp);
  *sizep = size;
  return cp;
}

// Read and encode input file and return malloc'ed encoded one
// (+length). Caller must free returned buffer
char *base64EncodeFile(const char *fname, int *outlen)
{
  char *cp, *cp1;
  int size;

  if ((cp = readFile(fname, &size)) == 0)
	return 0;
  cp1 = base64Encode(cp, size, outlen);

  free(cp);
  return cp1;
}

//////////////////////////////////////////////

// Internal copy from peer function. We supposed we've been reset by the
// caller (no freeing)
void 
HeaderFieldParam::buildFromFriend(const HeaderFieldParam& rh) 
{
  m_name = m_value = 0;
  m_size = rh.m_size;
  if (rh.m_name)
	m_name = strdup(rh.m_name);
  if (rh.m_value)
	m_value = strdup(rh.m_value);
  if (m_name == 0 || m_value == 0)
	m_size = 0;
}


HeaderFieldParam& 
HeaderFieldParam::operator=(const HeaderFieldParam& l) 
{
  if (this == &l)
	return *this;
  reset();
  buildFromFriend(l);
  return *this;
}

// returns the size -without any terminating 0- Print() WILL overshoot
// size by 1
int 
HeaderFieldParam::size() 
{
  if (m_size == 0) {
	if (m_name == 0 || m_value == 0)
	  return -1;
	// 5 is ';', ' ', '=', '"', '"' (; filename="toto")
	m_size = strlen(m_name) + strlen(m_value) + 5;
  }
  return m_size;
}

int 
HeaderFieldParam::print(char *d) 
{
  if (size () < 0)
	return -1;
  sprintf(d, "; %s=\"%s\"", m_name, m_value);
  return size();
}

// Header field, like 'Content-Disposition: attachment; filename="toto"'

void 
HeaderField::buildFromFriend(const HeaderField &rh) 
{
  m_size = rh.m_size;
  m_name = strdup(rh.m_name);
  m_value = strdup(rh.m_value);
  if (m_name == 0 || m_value == 0)
	m_size = 0;
  for (list<HeaderFieldParam>::const_iterator it = rh.m_params.begin();
	   it != rh.m_params.end(); it++) {
	m_params.push_back(*it);
  }
}

HeaderField& HeaderField::operator=(const HeaderField &rh) 
{
  if (this == &rh)
	return *this;
  reset();
  buildFromFriend(rh);
  return *this;
}

int 
HeaderField::addparam(const char *pnm, const char *pval) 
{
  m_size = 0;
  m_params.push_back(HeaderFieldParam(pnm, pval));
  return 0;
}

// Return size, -without any terminating 0- print() WILL overshoot size by 1
int 
HeaderField::size() 
{
  if (m_size == 0) {
	if (m_name == 0 || m_value == 0) 
	  return -1;
	// 4 is for ': ' + final crlf
	m_size = 4 + strlen(m_name) + strlen(m_value);
	for (list<HeaderFieldParam>::iterator i = m_params.begin(); 
		 i != m_params.end(); i++) {
	  int s;
	  if ((s = i->size()) < 0)
		return -1;
	  m_size += s;
	}
  }
  return m_size;
}

int 
HeaderField::print(char *d) 
{
  char *od = d;
  if (size() < 0)
	return -1;
  sprintf(d, "%s: %s", m_name, m_value);
  // 2 is for ": "
  d += strlen(m_name) + strlen(m_value) + 2;
  // Parameters
  for (list<HeaderFieldParam>::iterator i = m_params.begin(); 
	   i != m_params.end(); i++) {
	if (i->size() < 0)
	  return -1;
	d += i->print(d);
  }
  *d++ = '\r';
  *d++ = '\n';
  *d = 0;
  //fprintf(stderr, "HeaderField::print: size() %d, actual %d\n",size(),d-od);
  return d - od;
}


////////////////////////////////////////////////////////

void 
Entity::seterror(const char *s) 
{
  freeZ(m_error);
  m_error = strdup(s);
}

int 
Entity::setsubtype(const char *stp) 
{
  freeZ(m_subtype);
  m_subtype = strdup(stp);
  if (m_subtype == 0)
	return -1;
  return 0;
}

void 
Entity::addHeaderField(const HeaderField &f) 
{
  m_hf.push_back(f);
  m_size = 0;
}

int 
Entity::allocFmt() 
{
  freeZ(m_fmt);
  if ((m_fmt = (char *)malloc(size() + 1)) == 0) {
	seterror("Out of memory");
	return -1;
  }
  return 0;
}

int 
Entity::formatHeaders(char *buf) 
{
  char *cp = buf;
  for (list<HeaderField>::iterator it = m_hf.begin(); 
	   it != m_hf.end(); it++) {
	if (it->size() < 0) {
	  seterror("Bad Header field");
	  return -1;
	}
	cp += it->print(cp);
  }
  return cp - buf;
}

int 
Entity::addCteHeader()
{
  char *cenc = "binary";
  switch (m_encoding) {
  case ENC_7BIT: cenc = "7bit";break;
  case ENC_8BIT:cenc = "8bit";break;
  case ENC_BINARY:cenc = "binary";break;
  case ENC_QP:cenc = "quoted-printable";break;
  case ENC_BASE64:cenc = "base64";break;
  default:break;
  }

  addHeaderField(HeaderField("Content-Transfer-Encoding", cenc));
  return 0;
}

int 
Entity::formatHeaders() 
{
  if (m_fmt == 0) {
	seterror("Internal logic error: formatHeaders called before alloc");
	return -1;
  }
  return formatHeaders(m_fmt);
}


///////////////////////////////////////////

void 
DiscreteEntity::setTextEncodePref(encoding ec) 
{
  switch (ec) {
  case ENC_8BIT: 
  case ENC_QP: 
	m_textencodepref = ec;
	break;
  default: break;
  }
}

int DiscreteEntity::addCtpHeader()
{
  char *ctp = "unknown";
  switch (m_type) {
  case TEXT: ctp = "text";break;
  case IMAGE:ctp = "image";break;
  case AUDIO:ctp = "audio";break;
  case VIDEO:ctp = "video";break;
  case APPLICATION:ctp = "application";break;
  default:break;
  }
  string t = ctp;
  t += "/";
  t += getsubtype();
  HeaderField hf("Content-Type", t.c_str());
  if (m_type == TEXT && m_charset.size())
	hf.addparam("charset", m_charset.c_str());
  addHeaderField(hf);
  return addCteHeader();
}

int 
DiscreteEntity::setBody(const char *bp, int blen, type tp, const char *stp) 
{
  freeZ(m_body);
  settype(tp);
  if (setsubtype(stp) < 0)
	return -1;

  switch (tp) {
  case TEXT:
	{
	  // Choose encoding for message.
	  // We test for null bytes, 8bit characters, and lines exceeding
	  // 998 chars. We also count lf characters that would need to be
	  // changed to crlf, then depending on the results and user
	  // prefs, we qp-encode or not
	  int has8bit = 0;
	  int hasnull = 0;
	  int nlcnt = 0; // *Isolated* nl *or* cr count
	  int maxline = 0;
	  int curline = 0;
	  for (int i = 0; i < blen; i++) {
		unsigned char c = (unsigned char)(bp[i]);
		if (c == 0)
		  hasnull = 1;
		if (c > 127)
		  has8bit = 1;

		// Line length check
		if (c == '\r' || c == '\n') {
		  if (curline > maxline)
			maxline = curline;
		  curline = 0;
		} else {
		  ++curline;
		}

		// cr->crlf, lf->crlf checking
		if (c == '\r') {
		  if (i == blen - 1) {
			nlcnt++;
		  } else {
			if (bp[i+1] == '\n') {
			  i++;
			  continue;
			} else {
			  nlcnt++;
			}
		  }
		} else if (c == '\n') {
		  nlcnt++;
		}
	  }

	  //fprintf(stderr, "maxline %d hasnull %d has8bit %d nlcnt %d\n",
	  //			  maxline, hasnull, has8bit, nlcnt);

	  if (hasnull || maxline >= 998) {
		m_encoding = ENC_QP;
	  } else {
		m_encoding = has8bit ? m_textencodepref : ENC_7BIT;
	  }

	  if (m_charset.empty()) 
		m_charset = has8bit ? "iso8859-1" : "us-ascii";

	  if (m_encoding == ENC_7BIT || m_encoding == ENC_8BIT) {
		// No encoding. But we do fix end of line sequences. and possibly
		// add a final crlf
		// We allocate the 2 bytes for possible crlf at eot in all cases
		//       base   lf->crlf   final crlf     final 0
		m_blen = blen + nlcnt    + 2              + 1;
		if ((m_body = (char *)malloc(m_blen)) == 0) {
		  seterror("Out of memory");
		  return 0;
		}

		// Copy input to output, just fixing line terminations
		char *cp = m_body;
		for (int i = 0; i < blen; i++) {
		  if (bp[i] == '\r') {
			*cp++ = '\r';
			*cp++ = '\n';
			if (i < blen - 1 && bp[i+1] == '\n') {
			  ++i;
			}
			continue;
		  }
		  if (bp[i] == '\n') {
			*cp++ = '\r';
			*cp++ = '\n';
			continue;
		  }
		  *cp++ = bp[i];
		}

		// Need final crlf ?
		if (cp - m_body < 2 || cp[-1] != '\n') {
		  *cp++ = '\r';
		  *cp++ = '\n';
		}

		// Note: final 0 not included in blen
		m_blen = cp - m_body;
		*cp++ = 0;

	  } else {
		// Quoted printable
		m_body = qpEncode(bp, blen, &m_blen);
	  }
	}
	break;
  default: 
	{
	  m_encoding = ENC_BASE64;
	  m_body = base64Encode(bp, blen, &m_blen);
	}
	break;
  }
  
  return addCtpHeader();
}

int DiscreteEntity::setBodyAttach(const char *filename, type tp,
								  const char *stp) 
{
  freeZ(m_body);
  settype(tp);
  if (setsubtype(stp) < 0)
	return -1;
  m_encoding = ENC_BASE64;
  m_body = base64EncodeFile(filename, &m_blen);
  if (m_body == 0)
	return -1;
  HeaderField hf("Content-Disposition", "attachment");
  char *cp = strrchr(filename, '/');
  hf.addparam("filename", cp ? cp+1 : filename);
  addHeaderField(hf);
  return addCtpHeader();
}

int 
DiscreteEntity::size() 
{
  if (m_size == 0) {
	for (list<HeaderField>::iterator it = m_hf.begin(); 
		 it != m_hf.end(); it++) {
	  if (it->size() < 0)
		return -1;
	  m_size += it->size();
	}
	// Empty line (crlf)
	m_size += 2;
	// Body
	m_size += m_blen;
	// Final 0
	m_size++;
  }
	
  return m_size;
}

int 
DiscreteEntity::format(char *buf) 
{
  int hl = formatHeaders(buf);
  if (hl < 0) 
	return 0;
  char *cp = buf + hl;
  *cp++ = '\r'; 
  *cp++ = '\n';
  memcpy(cp, m_body, m_blen);
  cp += m_blen;
  *cp++ = 0;
  return cp - buf;
}

const char *
DiscreteEntity::format() 
{
  if (allocFmt() < 0)
	return 0;
  if (format(m_fmt) < 0)
	return 0;
  return m_fmt;
}

////////////////////////////////////////////////////////////
CompositeEntity::CompositeEntity(type tp, const char *stp) : m_size(0) 
{
  m_encoding = ENC_7BIT;
  settype(tp);
  setsubtype(stp);
  string t = "multipart/";
  t += getsubtype();
  HeaderField hf("Content-Type",t.c_str());
  m_boundary = "_X=------=_NextPart_";
  char c[20];
  sprintf(c, "%x%x%x", (unsigned int)this, (unsigned int)time(0),
#if defined(unix)
	  (unsigned int)getpid()
#else
	  0
#endif 
	  );

  m_boundary += c;
  hf.addparam("boundary", m_boundary.c_str());
  addHeaderField(hf);
}

CompositeEntity::~CompositeEntity() 
{
  for (list<Entity*>::iterator i = m_parts.begin(); i != m_parts.end();i++)
	delete *i;
}

int 
CompositeEntity::addPart(DiscreteEntity *part) 
{
  m_parts.push_back(part);
  encoding penc = part->getencoding();
  if (penc != ENC_7BIT && penc != ENC_QP && penc != ENC_BASE64)
	m_encoding =  ENC_8BIT;
  m_size = 0;
  return 0;
}

int 
CompositeEntity::size() 
{
  if (m_type != MULTIPART) {
	seterror("Only MULTIPART supported for now");
	return -1;
  }

  // Add transfer encoding header
  // There is a problem here: we'd like size() not to change the
  // message, but we also need to be sure that the transfer encoding
  // header is set somewhere, which must be done after all parts are
  // added. This might result in several transfer-encoding headers being 
  // added, which seems less severe than the previous situation where the 
  // size() was computed without this header...
  addCteHeader();

  if (m_size == 0) {
	// Headers
	for (list<HeaderField>::iterator it = m_hf.begin(); 
		 it != m_hf.end(); it++) {
	  if (it->size() < 0)
		return -1;
	  m_size += it->size();
	}

	// Empty line (crlf)
	m_size += 2;

	// Body parts
	list<Entity *>::iterator itE = m_parts.begin();
	for (;itE != m_parts.end(); itE++) {
	  m_size += m_boundary.size() + 6; // 2 -- + 2 crlf
	  // it->size() includes final 0
	  m_size += (*itE)->size() - 1;
	}
	// Final boundary: 4*'-' + 2*crlf
	m_size += m_boundary.size() + 8;
	// Final 0
	m_size++;
  }
  
  return m_size;
}

int 
CompositeEntity::format(char *buf)
{
  int hl = formatHeaders();
  if (hl < 0) 
	return 0;
  char *cp = buf + hl;
  *cp++ = '\r'; 
  *cp++ = '\n';

  for (list<Entity *>::iterator it = m_parts.begin();
	   it != m_parts.end(); it++) {
	*cp++ = '\r';*cp++ = '\n';*cp++= '-';*cp++= '-';
	memcpy(cp, m_boundary.c_str(), m_boundary.size());
	cp += m_boundary.size();
	*cp++ = '\r'; *cp++ = '\n';
	cp += (*it)->format(cp);
	// Get rid of final 0
	cp--;
  }

  // Final boundary
  *cp++ = '\r'; *cp++ = '\n'; *cp++= '-'; *cp++= '-';
  memcpy(cp, m_boundary.c_str(), m_boundary.size());
  cp += m_boundary.size();
  *cp++= '-'; *cp++= '-'; *cp++ = '\r'; *cp++ = '\n';
	  
  // final 0
  *cp++ = 0;
  //  fprintf(stderr, "CompositeEntity::format: size() %d, actual %d\n",
  //		  size(), cp - buf);
  return cp - buf;
}

const char *
CompositeEntity::format() 
{
  if (allocFmt() < 0)
	return 0;
  if (format(m_fmt) < 0)
	return 0;
  return m_fmt;
}

////////////////////////////////////////////////////////////////

const char *WriteMime::Message::recpth[3] = {"To", "Cc", "Bcc"};

Message::Message() 
{
  recptlists[TO] = &to;
  recptlists[CC] = &cc;
  recptlists[BCC] = &bcc;
}

void 
Message::setRecipientHeaders() 
{
  // Note that printing out bcc would rather defeat the intent...
  for (int i = 0; i < 2; i++) {
	string dests;
	dests.erase();
	for (list<Recipient>::iterator it = recptlists[i]->begin();
		 it != recptlists[i]->end();it++) {
	  if (it != recptlists[i]->begin())
		dests += ", ";
	  if (!it->nickname.empty()) {
		dests += it->nickname;
		dests += " ";
	  }
	  if (!it->official.empty()) {
		dests += "<";
		dests += it->official;
		dests += ">";
	  }
	}
	if (dests.size())
	  addHeaderField(HeaderField(recpth[i], dests.c_str()));
  }
}

// Add recipient to one of the recipient lists
int Message::addRecipient(const char *official, const char *nick, 
						  recpt_type tp)
{
  if (tp < 0 || tp > 3)
	return -1;
  recptlists[tp]->push_back(Recipient(official, nick));
  return 0;
}

// Parse recipient string and add recipients to appriopriate list
// A recipient list is a comma separated list of recipients. Parsing it is not
// trivial because of quoting and commenting. Ie one element may be like:
// "mon,ami" (un commentaire, quoi) <monami@somesite.com>
enum parseRecptsState {PST_BASE, PST_QUOTE, PST_COMMENT, PST_OFFICIAL};
int Message::parseAddRecipients(const char *s, recpt_type tp) 
{
  string nick;
  string off;
  int c;
  parseRecptsState state = PST_BASE;
  parseRecptsState prevstate = state;
  int l = strlen(s);

  for (int i = 0; i < l;i++) {
	switch(c = s[i]) {
	case ',': 
	  // fprintf(stderr, "Nick '%s', Off: '%s'\n", nick.c_str(), off.c_str());
	  if (!nick.empty() || !off.empty())
		addRecipient(nick.c_str(), off.c_str(), tp);
	  nick.erase(); 
	  off.erase();
	  break;

	case '(': 
	  switch (state) {
	  case PST_QUOTE:
		nick += c;
		break;
	  default:
		if (state != PST_COMMENT) {
		  prevstate = state; 
		  state = PST_COMMENT;
		}
	  }
	  break;

	case ')': 
	  switch (state) {
	  case PST_QUOTE:
		nick += c;
		break;
	  default:
		if (state != PST_COMMENT) {
		  seterror("Comment close while not in comment\n");
		  return -1;
		}
		state = prevstate;
	  }
	  break;

	case '<': 
	  switch (state) {
	  case PST_BASE: 
		state = PST_OFFICIAL;
		break;
	  case PST_OFFICIAL: 
		seterror("Bad character '<' while in canonic address mode\n");
		return -1;
	  case PST_QUOTE: 
		nick += c;
		break;
	  case PST_COMMENT:
		break;
	  }
	  break;

	case '>':
	  switch(state) {
	  case PST_BASE: 
		seterror("Unquoted '>' character while in base mode\n");
		return -1;
	  case PST_OFFICIAL: 
		state = PST_BASE;
		break;
	  case PST_QUOTE: 
		nick += c;
		break;
	  case PST_COMMENT: 
		break;
	  }
	  break;

	case '"': 
	  switch (state) {
	  case PST_BASE:
		state = PST_QUOTE;
		break;
	  case PST_OFFICIAL:
		seterror("Quote found inside canonic address field\n");
		return -1;
	  case PST_QUOTE:
		state = PST_BASE;
		break;
	  case PST_COMMENT:
		break;

	  }
	  break;

	default: 
	  switch (state) {
	  case PST_QUOTE:
	  case PST_BASE: 
		nick += c; 
		break;
	  case PST_OFFICIAL: 
		off += c;
		break;
	  case PST_COMMENT:
		break;
	  }
	  break;
	}
	
  }
//  fprintf(stderr, "Nick '%s', Off: '%s'\n", nick.c_str(), off.c_str());
  if (!nick.empty() || !off.empty())
	addRecipient(off.c_str(), nick.c_str(), tp);
  return 0;
}

// Return recipient list in a form suitable for program execution
const char ** Message::getRecipients() 
{
  int n = 0;
  for (int i = 0; i < 3; i++) 
	n += recptlists[i]->size();
  n++; // final 0

  const char ** v = (const char **)malloc(n * sizeof(char *));

  n = 0;
  for (int j = 0; j < 3; j++) {
	for (list<Recipient>::iterator it = recptlists[j]->begin();
		 it != recptlists[j]->end();it++) {
	  if (!it->official.empty())
		v[n++]= it->official.c_str();
	  else 
		v[n++]= it->nickname.c_str();
	}
  }
  v[n] = 0;
  return v;
}

int 
Message::setSubject(const char *subject) 
{
  addHeaderField(HeaderField("Subject", subject));
  return 0;
}

int 
Message::setFrom(const Recipient &from)
{
  string fromstring = from.nickname;
  fromstring += " <";
  fromstring += from.official;
  fromstring += ">";
  addHeaderField(HeaderField("From", fromstring.c_str()));
  return 0;
}

///////////////////////////////////////////////////////
SimpleMessage::SimpleMessage() 
{
  addHeaderField(HeaderField("MIME-Version", "1.0"));
}	

// return 0-terminated formatted message text
const char *
SimpleMessage::format() 
{
  setRecipientHeaders();
  return DiscreteEntity::format();
}

///////////////////////////////////////////////////////

CompositeMessage::CompositeMessage(type tp, const char *stp) 
  : CompositeEntity(tp, stp) 
{
  addHeaderField(HeaderField("MIME-Version", "1.0"));
}	

// return 0-terminated formatted message text
const char *
CompositeMessage::format() 
{
  setRecipientHeaders();
  return CompositeEntity::format();
}

/////////////////////////////////////////////////////////

#ifdef WITH_SENDMAIL

// Default value for the sendmail program file path. This can be overridden
// by the  environment variable
#ifndef SENDMAIL_PROG
#define SENDMAIL_PROG "/usr/sbin/sendmail"
#endif

int 
sendmail(Message *msg)
{
  const char **recpt = msg->getRecipients();

  int nrcpt = 0;
  for (nrcpt = 0;;nrcpt++)
	if (recpt[nrcpt] == 0)
	  break;
  if (nrcpt == 0) {
	msg->seterror("No recipients");
	return -1;
  }
  const char *msgcp = msg->format();

  if (msgcp == 0) {
	// fprintf(stderr, "Error: %s\n", msg->geterror());
	return -1;
  }
  //  printf("%s", msgcp);

  char **argv = (char **)malloc((nrcpt + 4)*sizeof(char *));
  char *ep = getenv("WRITEMIME_SENDMAIL_PROG");
  if (ep)
	argv[0] = ep;
  else 
	argv[0] = SENDMAIL_PROG;
  argv[1] = "-i";
  argv[2] = "--";
  for (int i = 3; i < nrcpt+3; i++)
	argv[i] = (char *)recpt[i-3];
  argv[nrcpt+3] = 0;

  int fd[2];
  if (pipe(fd) < 0) {
	msg->seterror("System error: pipe");
	return -1;
  }
  int pid = fork();
  if (pid == 0) {
	// In son process
	close(fd[1]);
	if (fd[0] != 0) {
	  close(0);
	  dup(fd[0]);
	  close(fd[0]);
	}

#if 0
	fprintf(stderr, "executing: ");
	for (int i = 0;;i++) {
	  if (argv[i] == 0)
		break;
	  fprintf(stderr, "%s ", argv[i]); 
	}
	fprintf(stderr, "\n");
#endif 

	execv(SENDMAIL_PROG, (char *const*)argv);
	perror("execv");
	exit(1);
  } else if (pid > 0) {
	// In father
	struct sigaction sa, osa;
	sa.sa_flags = 0;
	sigemptyset(&sa.sa_mask);
	sa.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &sa, &osa);
	close(fd[0]);
	int l = strlen(msgcp);
	if (write(fd[1], msgcp, l) != l) {
	  msg->seterror("cannot write to sendmail process");
	  sigaction(SIGPIPE, &osa, 0);
	  return -1;
	}
	close(fd[1]);
	sigaction(SIGPIPE, &osa, 0);
	int status;
	waitpid(pid, &status, 0);
	if (status) {
	  char error[100];
	  sprintf(error,"sendmail exit status: 0x%x\n", status); 
	  msg->seterror(error);
	  return -1;
	}
	return 0;
  } else {
	msg->seterror("System error: cant fork");
	return -1;
  }
  // cant get there but anyway
  return -1;
}

#endif /* WITH_SENDMAIL */

#ifndef NO_NAMESPACES
} // end namespace WriteMime
#endif // NO_NAMESPACES
