/*
  Copyright (c) 2009 Jean-Francois Dockes

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following
  conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
  OTHER DEALINGS IN THE SOFTWARE.
*/

#include "pxattr.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <ftw.h>
#include <sys/types.h>
#include <regex.h>

#include <iostream>
#include <fstream>
#include <map>
#include <algorithm>
#include <string>

static int antiverbose;

static void printsyserr(const std::string& msg)
{
    if (antiverbose >= 2)
        return;
    std::cerr << msg << " " << strerror(errno) << '\n';
}

#define message(X)                              \
    {                                           \
        if (antiverbose == 0) {                 \
            std::cout << X;                     \
        }                                       \
    }
    
static void dotests();

// \-quote character qc in input \ -> \\, nl -> \n cr -> \rc -> \c
static void quote(const std::string& in, std::string& out, int qc)
{
    out.clear();
    for (const auto c : in) {
        if (c == '\\') {
            out += "\\\\";
        } else if (c == "\n"[0]) {
            out += "\\n";
        } else if (c == "\r"[0]) {
            out += "\\r";
        } else if (c == qc) {
            out += "\\";
            out += c;
        } else {
            out += c;
        }
    }
}

// \-unquote input \n -> nl, \r -> cr, \c -> c
static void unquote(const std::string& in, std::string& out)
{
    out.clear();
    for (unsigned int i = 0; i < in.size(); i++) {
        if (in[i] == '\\') {
            if (i == in.size() -1) {
                out += in[i];
            } else {
                int c = in[++i];
                switch (c) {
                case 'n': out += "\n";break;
                case 'r': out += "\r";break;
                default: out += c;
                }
            }
        } else {
            out += in[i];
        }
    }
}

// Find first unquoted c in input: c preceded by odd number of backslashes
std::string::size_type find_first_unquoted(const std::string& in, int c)
{
    int q = 0;
    for (unsigned int i = 0;i < in.size(); i++) {
        if (in[i] == '\\') {
            q++;
        } else if (in[i] == c) {
            if (q&1) {
                // quoted
                q = 0;
            } else {
                return i;
            }
        } else {
            q = 0;
        }
    }
    return std::string::npos;
}

static const std::string PATH_START("Path: ");
static bool listattrs(const std::string& path)
{
    std::vector<std::string> names;
    if (!pxattr::list(path, &names)) {
        if (errno == ENOENT) {
            return false;
        }
        printsyserr("pxattr::list");
        exit(1);
    }
    if (names.empty()) {
        return true;
    }

    // Sorting the names would not be necessary but it makes easier comparing
    // backups
    sort(names.begin(), names.end());

    std::string quoted;
    quote(path, quoted, 0);
    message(PATH_START << quoted << '\n');
    for (const auto nm : names) {
#ifdef __APPLE__
        static const std::string applekeystart("com.apple.");
        if (nm.compare(0, applekeystart.size(), applekeystart) == 0) {
            // Can't process these usefully
            std::cerr << "Skipping apple finder key " << nm << '\n';
            continue;
        }
#endif // __APPLE__
        
        std::string value;
        if (!pxattr::get(path, nm, &value)) {
            if (errno == ENOENT) {
                return false;
            }
            printsyserr("pxattr::get");
            exit(1);
        }
        quote(nm, quoted, '=');
        message(" " << quoted << "=");
        quote(value, quoted, 0);
        message(quoted << '\n');
    }
    return true;
}

bool setxattr(const std::string& path, const std::string& name, const std::string& value)
{
    if (!pxattr::set(path, name, value)) {
        printsyserr("pxattr::set");
        return false;
    }
    return true;
}

#ifdef __APPLE__
#include "finderxattr.h"
#include <iconv.h>

bool transcode(
    const std::string &in, std::string &out, const std::string& icode, const std::string& ocode)
{
    bool ret = false;
    const int OBSIZ = 8192;
    char obuf[OBSIZ];
    iconv_t ic = (iconv_t)-1;
    const char *ip = in.c_str();
    size_t isiz = in.length();
    out.erase();
    out.reserve(isiz);

    if((ic = iconv_open(ocode.c_str(), icode.c_str())) == (iconv_t)-1) {
        out = std::string("iconv_open failed for ") + icode  + " -> " + ocode;
        goto error;
    }

    while (isiz > 0) {
        size_t osiz;
        char *op = obuf;
        osiz = OBSIZ;
        if (iconv(ic, (char **)&ip, &isiz, &op, &osiz) == (size_t)-1 && errno != E2BIG) {
            if (errno == EILSEQ) {
                out.append(obuf, OBSIZ - osiz);
                out += "?"; ip++; isiz--;
                continue;
            }
            if (errno == EINVAL)
                goto out;
            goto error;
        }
        out.append(obuf, OBSIZ - osiz);
    }
out:
    ret = true;
error:
    if (ic != (iconv_t)-1) {
        iconv_close(ic);
    }
    return ret;
}

std::string decode_finder_metadata(
    const std::string& path, const std::string& xkey, const std::string& value)
{
    if (xkey == "com.apple.metadata:kMDItemFinderComment") {
        auto [v1, error] = decode_comment_plist(
            (const unsigned char *)value.c_str(), (int)value.size());
        if (!error.empty()) {
            std::cerr << "Failed decoding finder comment for " << path << "\n";
            return value;
        } else {
            if (!v1.empty() && v1[0] == 0) {
                std::string v2;
                transcode(v1, v2, "UCS-2BE", "UTF-8"); return v2;
                return v2;
            } else {
                return v1;
            }
        }
    } else if (xkey == "com.apple.metadata:_kMDItemUserTags") {
        auto [v1, error] = decode_tags_plist(
            (const unsigned char *)value.c_str(), (int)value.size());
        if (!error.empty()) {
            std::cerr << "Failed decoding finder tags for " << path << "\n";
            return value;
        } else if (v1.empty()) {
            return std::string();
        } else {
            // The tags have a color number at the end, which we discard as not interesting.
            std::string nvalue;
            for (auto& _s : v1) {
                std::string s;
                if (!_s.empty() && _s[0] == 0) {
                    transcode(_s, s, "UCS-2BE", "UTF-8");
                } else {
                    _s.swap(s);
                }
                auto pos = s.find_last_of(" \t\n\r");
                if (pos != std::string::npos && pos < s.size() - 1) {
                    if (isdigit(s[pos+1])) {
                        s.erase(pos);
                    }
                }
                nvalue += s + " ";
            }
            return nvalue;
        }
    }
    // Other attributes: no decoding
    return value;
}
#endif // __APPLE__

bool printxattr(const std::string &path, const std::string& name)
{
    std::string value;
    if (!pxattr::get(path, name, &value)) {
        if (errno == ENOENT) {
            return false;
        }
        printsyserr("pxattr::get");
        return false;
    }
    message(PATH_START << path << '\n');
#ifdef __APPLE__
    value = decode_finder_metadata(path, name, value);
#endif // __APPLE__
    message(" " << name << " => " << value << '\n');
    return true;
}

bool delxattr(const std::string &path, const std::string& name) 
{
    if (!pxattr::del(path, name)) {
        printsyserr("pxattr::del");
        return false;
    }
    return true;
}

// Restore xattrs stored in file created by pxattr -lR output
static void restore(const char *backupnm)
{
    std::istream *input;
    std::ifstream fin;
    if (!strcmp(backupnm, "stdin")) {
        input = &std::cin;
    } else {
        fin.open(backupnm, std::ios::in);
        input = &fin;
    }

    bool done = false;
    int linenum = 0;
    std::string path;
    std::map<std::string, std::string> attrs;
    while (!done) {
        std::string line;
        getline(*input, line);
        if (!input->good()) {
            if (input->bad()) {
                std::cerr << "Input I/O error" << '\n';
                exit(1);
            }
            done = true;
        } else {
            linenum++;
        }

        // message("Got line " << linenum << " : [" << line << "] done " << 
        // done << '\n');

        if (line.find(PATH_START) == 0 || done) {
            if (!path.empty() && !attrs.empty()) {
                for (const auto& [nm, value] : attrs) {
                    setxattr(path, nm, value);
                }
            }
            if (!done) {
                line = line.substr(PATH_START.size(), std::string::npos);
                unquote(line, path);
                attrs.clear();
            }
        } else if (line.empty()) {
            continue;
        } else {
            // Should be attribute line
            if (line[0] != ' ') {
                std::cerr << "Found bad line (no space) at " << linenum << '\n';
                exit(1);
            }
            std::string::size_type pos = find_first_unquoted(line, '=');
            if (pos == std::string::npos || pos < 2 || pos >= line.size()) {
                std::cerr << "Found bad line at " << linenum << '\n';
                exit(1);
            }
            std::string qname = line.substr(1, pos-1);
            std::pair<std::string,std::string> entry;
            unquote(qname, entry.first);
            unquote(line.substr(pos+1), entry.second);
            attrs.insert(entry);
        }
    }
}

static char *thisprog;
static char usage [] =
    "pxattr [-hs] -n name pathname [...] : show value for name\n"
    "pxattr [-hs] -n name -r regexp pathname [...] : test value against regexp\n"
    "pxattr [-hs] -n name -v value pathname [...] : add/replace attribute\n"
    "pxattr [-hs] -x name pathname [...] : delete attribute\n"
    "pxattr [-hs] [-l] [-R] pathname [...] : list attribute names and values\n"
    "  For all the options above, if no pathname arguments are given, pxattr\n"
    "  will read file names on stdin, one per line.\n"
    " [-h] : don't follow symbolic links (act on link itself)\n"
    " [-R] : recursive listing. Args should be directory(ies)\n"
    " [-s] : be silent. With one option stdout is suppressed, with 2 stderr too\n"
    "pxattr -S <backupfile> Restore xattrs from file created by pxattr -lR output\n"
    "               if backupfile is 'stdin', reads from stdin\n"
    "pxattr -T: run tests on temp file in current directory" 
    "\n"
    ;
static void Usage(void)
{
    fprintf(stderr, "%s: usage:\n%s", thisprog, usage);
    exit(1);
}

static int     op_flags;
#define OPT_MOINS 0x1
#define OPT_h     0x2
#define OPT_l     0x4
#define OPT_n     0x8
#define OPT_r     0x10
#define OPT_R     0x20
#define OPT_S     0x40
#define OPT_s     0x80
#define OPT_T     0x100
#define OPT_v     0x200
#define OPT_x     0x400

// Static values for ftw
static std::string name, value;

bool regex_test(const char *path, regex_t *preg)
{
    std::string value;
    if (!pxattr::get(path, name, &value)) {
        if (errno == ENOENT) {
            return false;
        }
        printsyserr("pxattr::get");
        return false;
    }

    int ret = regexec(preg, value.c_str(), 0, 0, 0);
    if (ret == 0) {
        message(path << '\n');
        return true;
    } else if (ret == REG_NOMATCH) {
        return false;
    } else {
        char errmsg[200];
        regerror(ret, preg, errmsg, 200);
        errno = 0;
        printsyserr("regexec");
        return false;
    }
}

bool processfile(const char* fn, const struct stat *, int)
{
    //message("processfile " << fn << " opflags " << op_flags << '\n');

    if (op_flags & OPT_l) {
        return listattrs(fn);
    } else if (op_flags & OPT_n) {
        if (op_flags & OPT_v) {
            return setxattr(fn, name, value);
        } else {
            return printxattr(fn, name);
        } 
    } else if (op_flags & OPT_x) {
        return delxattr(fn, name);
    }
    Usage();
    return false;
}

int ftwprocessfile(const char* fn, const struct stat *sb, int typeflag)
{
    processfile(fn, sb, typeflag);
    return 0;
}

int main(int argc, char **argv)
{
    const char *regexp_string;
    thisprog = argv[0];
    argc--; argv++;
    
    while (argc > 0 && **argv == '-') {
        (*argv)++;
        if (!(**argv))
            /* Cas du "adb - core" */
            Usage();
        while (**argv)
            switch (*(*argv)++) {
            case 'l':    op_flags |= OPT_l; break;
            case 'n':    op_flags |= OPT_n; if (argc < 2)  Usage();
                name = *(++argv); argc--; 
                goto b1;
            case 'R':    op_flags |= OPT_R; break;
            case 'r':    op_flags |= OPT_r; if (argc < 2)  Usage();
                regexp_string = *(++argv); argc--; 
                goto b1;
            case 's':    antiverbose++; break;
            case 'S':    op_flags |= OPT_S; break;
            case 'T':    op_flags |= OPT_T; break;
            case 'v':    op_flags |= OPT_v; if (argc < 2)  Usage();
                value = *(++argv); argc--; 
                goto b1;
            case 'x':    op_flags |= OPT_x; if (argc < 2)  Usage();
                name = *(++argv); argc--; 
                goto b1;
            default: Usage();    break;
            }
    b1: argc--; argv++;
    }

    if (op_flags & OPT_T)  {
        if (argc > 0)
            Usage();
        dotests();
        exit(0);
    }
    if ((op_flags & OPT_r) && !(op_flags & OPT_n)) {
        Usage();
    }
    
    if (op_flags & OPT_S)  {
        if (argc != 1)
            Usage();
        restore(argv[0]);
        exit(0);
    }
    regex_t regexp;
    if (op_flags & OPT_r) {
        int err = regcomp(&regexp, regexp_string, REG_NOSUB|REG_EXTENDED);
        if (err) {
            char errmsg[200];
            regerror(err, &regexp, errmsg, 200);
            std::cerr << "regcomp(" << regexp_string << ") error: " << errmsg << '\n';
            exit(1);
        }
    }
    
    // Default option is 'list'
    if ((op_flags&(OPT_l|OPT_n|OPT_x)) == 0)
        op_flags |= OPT_l;

    bool readstdin = false;
    if (argc == 0)
        readstdin = true;

    int exitvalue = 0;
    for (;;) {
        const char *fn = 0;
        if (argc > 0) {
            fn = *argv++; 
            argc--;
        } else if (readstdin) {
            static char filename[1025];
            if (!fgets(filename, 1024, stdin))
                break;
            filename[strlen(filename)-1] = 0;
            fn = filename;
        } else
            break;

        if (op_flags & OPT_R) {
            if (ftw(fn, ftwprocessfile, 20))
                exit(1);
        } else if (op_flags & OPT_r) {
            if (!regex_test(fn, &regexp)) {
                exitvalue = 1;
            }
        } else {
            if (!processfile(fn, 0, 0)) {
                exitvalue = 1;
            }
        }
    } 

    exit(exitvalue);
}

static void fatal(const std::string& s)
{
    printsyserr(s.c_str());
    exit(1);
}

static bool testbackups()
{
    message("Testing Backups\n");
    static const char *top = "ttop";
    static const char *d1 = "d1";
    static const char *d2 = "d2";
    static const char *tfn1 = "tpxattr1.txt";
    static const char *tfn2 = "tpxattr2.txt";
    static const char *dump = "attrdump.txt";
    static const char *NAMES[] = {"ORG.PXATTR.NAME1", 
                                  "ORG=PXATTR\"=\\=\n", 
                                  "=", "Name4"};
    static const char *VALUES[] = 
        {"VALUE1", "VALUE2", "VALUE3=VALUE3equal",
         "VALUE4\n is more like"
         " normal text\n with new lines and \"\\\" \\\" backslashes"};

    static const int nattrs = sizeof(NAMES) / sizeof(char *);

    if (mkdir(top, 0777))
        fatal("Cant mkdir ttop");
    if (chdir(top))
        fatal("cant chdir ttop");
    if (mkdir(d1, 0777) || mkdir(d2, 0777))
        fatal("Can't mkdir ttdop/dx\n");
    if (chdir(d1))
        fatal("chdir d1");

    int fd;
    if ((fd = open(tfn1, O_RDWR|O_CREAT, 0755)) < 0)
        fatal("create d1/tpxattr1.txt");
    /* Set attrs */
    for (int i = 0; i < nattrs; i++) {
        if (!pxattr::set(fd, NAMES[i], VALUES[i]))
            fatal("pxattr::set");
    }
    close(fd);
    if ((fd = open(tfn2, O_RDWR|O_CREAT, 0755)) < 0)
        fatal("create d1/tpxattr2.txt");
    /* Set attrs */
    for (int i = 0; i < nattrs; i++) {
        if (!pxattr::set(fd, NAMES[i], VALUES[i]))
            fatal("pxattr::set");
    }
    close(fd);

    /* Create dump */
    std::string cmd;
    cmd = std::string("pxattr -lR . > " ) + dump;
    if (system(cmd.c_str()))
        fatal(cmd + " in d1");
    if (chdir("../d2"))
        fatal("chdir ../d2");
    if (close(open(tfn1, O_RDWR|O_CREAT, 0755)))
        fatal("create d2/tpxattr.txt");
    if (close(open(tfn2, O_RDWR|O_CREAT, 0755)))
        fatal("create d2/tpxattr.txt");
    cmd = std::string("pxattr -S ../d1/" ) + dump;
    if (system(cmd.c_str()))
        fatal(cmd);
    cmd = std::string("pxattr -lR . > " ) + dump;
    if (system(cmd.c_str()))
        fatal(cmd + " in d2");
    cmd = std::string("diff ../d1/") + dump + " " + dump;
    if (system(cmd.c_str()))
        fatal(cmd);
    cmd = std::string("cat ") + dump;
    system(cmd.c_str());

    if (1) {
        unlink(dump);
        unlink(tfn1);
        unlink(tfn2);
        if (chdir("../d1"))
            fatal("chdir ../d1");
        unlink(dump);
        unlink(tfn1);
        unlink(tfn2);
        if (chdir("../"))
            fatal("chdir .. 1");
        if (rmdir(d1))
            fatal("rmdir d1");
        if (rmdir(d2))
            fatal("rmdir d2");
        if (chdir("../"))
            fatal("chdir .. 2");
        if (rmdir(top))
            fatal("rmdir ttop");
    }
    return true;
}

static void dotests()
{
    static const char *tfn = "pxattr_testtmp.xyz";
    static const char *NAMES[] = {"ORG.PXATTR.NAME1", "ORG.PXATTR.N2", 
                                  "ORG.PXATTR.LONGGGGGGGGisSSSHHHHHHHHHNAME3"};
    static const char *VALUES[] = {"VALUE1", "VALUE2", "VALUE3"};

    /* Create test file if it doesn't exist, remove all attributes */
    int fd = open(tfn, O_RDWR|O_CREAT, 0755);
    if (fd < 0) {
        printsyserr("open/create");
        exit(1);
    }

    if (!antiverbose)
        message("Cleanup old attrs\n");
    std::vector<std::string> names;
    if (!pxattr::list(tfn, &names)) {
        printsyserr("pxattr::list");
        exit(1);
    }
    for (std::vector<std::string>::const_iterator it = names.begin(); 
         it != names.end(); it++) {
        std::string value;
        if (!pxattr::del(fd, *it)) {
            printsyserr("pxattr::del");
            exit(1);
        }
    }
    /* Check that there are no attributes left */
    names.clear();
    if (!pxattr::list(tfn, &names)) {
        printsyserr("pxattr::list");
        exit(1);
    }
    if (names.size() != 0) {
        errno=0;printsyserr("Attributes remain after initial cleanup !\n");
        for (std::vector<std::string>::const_iterator it = names.begin();
             it != names.end(); it++) {
            if (antiverbose < 2)
                std::cerr << *it << '\n';
        }
        exit(1);
    }

    /* Create attributes, check existence and value */
    message("Creating extended attributes\n");
    for (int i = 0; i < 3; i++) {
        if (!pxattr::set(fd, NAMES[i], VALUES[i])) {
            printsyserr("pxattr::set");
            exit(1);
        }
    }
    message("Checking creation\n");
    for (int i = 0; i < 3; i++) {
        std::string value;
        if (!pxattr::get(tfn, NAMES[i], &value)) {
            printsyserr("pxattr::get");
            exit(1);
        }
        if (value.compare(VALUES[i])) {
            errno = 0;
            printsyserr("Wrong value after create !");
            exit(1);
        }
    }

    /* Delete one, check list */
    message("Delete one\n");
    if (!pxattr::del(tfn, NAMES[1])) {
        printsyserr("pxattr::del one name");
        exit(1);
    }
    message("Check list\n");
    for (int i = 0; i < 3; i++) {
        std::string value;
        if (!pxattr::get(fd, NAMES[i], &value)) {
            if (i == 1)
                continue;
            printsyserr("pxattr::get");
            exit(1);
        } else if (i == 1) {
            errno=0;
            printsyserr("Name at index 1 still exists after deletion\n");
            exit(1);
        }
        if (value.compare(VALUES[i])) {
            errno = 0;
            printsyserr("Wrong value after delete 1 !");
            exit(1);
        }
    }

    /* Test the CREATE/REPLACE flags */
    // Set existing with flag CREATE should fail
    message("Testing CREATE/REPLACE flags use\n");
    if (pxattr::set(tfn, NAMES[0], VALUES[0], pxattr::PXATTR_CREATE)) {
        errno=0;printsyserr("Create existing with flag CREATE succeeded !\n");
        exit(1);
    }
    // Set new with flag REPLACE should fail
    if (pxattr::set(tfn, NAMES[1], VALUES[1], pxattr::PXATTR_REPLACE)) {
        errno=0;printsyserr("Create new with flag REPLACE succeeded !\n");
        exit(1);
    }
    // Set new with flag CREATE should succeed
    if (!pxattr::set(fd, NAMES[1], VALUES[1], pxattr::PXATTR_CREATE)) {
        errno=0;printsyserr("Create new with flag CREATE failed !\n");
        exit(1);
    }
    // Set existing with flag REPLACE should succeed
    if (!pxattr::set(fd, NAMES[0], VALUES[0], pxattr::PXATTR_REPLACE)) {
        errno=0;printsyserr("Create existing with flag REPLACE failed !\n");
        exit(1);
    }
    close(fd);
    unlink(tfn);

    if (testbackups())
        exit(0);
    exit(1);
}
